document.addEventListener('DOMContentLoaded', function () {
    // 1. VERIFICA SE OS ELEMENTOS ESSENCIAIS EXISTEM
    const wrapper = document.getElementById('calendario-container-wrapper');
    const container = document.getElementById('calendario-container');
    const textoDiaSelecionado = document.getElementById('dia-selecionado-texto');
    const botaoComprar = document.getElementById('comprar-ingresso-btn');

    if (!wrapper || !container || !textoDiaSelecionado || !botaoComprar) {
        console.error('Um ou mais elementos essenciais do calendário não foram encontrados no HTML.');
        return; // Para a execução se o HTML base não estiver lá
    }

    // 2. DEFINIÇÃO DAS VARIÁVEIS DE ESTADO
    const urlBaseIngresso = calendarioConfig.urlIngresso || '';
    const apiUrl = calendarioConfig.apiUrl;
    const hoje = new Date();
    hoje.setHours(0, 0, 0, 0); 
    let dataAtual = new Date(hoje.getFullYear(), hoje.getMonth(), 1);
    let dataSelecionada = null;

    // 3. DEFINIÇÃO DE TODAS AS FUNÇÕES
    
    /**
     * Busca o preço para um único dia e atualiza o elemento no DOM.
     */
    async function buscarEExibirPreco(diaElemento, dataString) {
        const precoSpan = diaElemento.querySelector('.dia-preco');
        try {
            const response = await fetch(apiUrl + dataString);
            if (!response.ok) throw new Error('Sem preço');

            const data = await response.json();
            precoSpan.textContent = `R$${data.preco}`;
            diaElemento.classList.add('disponivel');
            diaElemento.addEventListener('click', () => selecionarDia(diaElemento.dataset.data));
        } catch (error) {
            diaElemento.classList.add('desabilitado');
            precoSpan.textContent = '';
        }
    }

    /**
     * Atualiza o rodapé com a data selecionada e o link de compra.
     */
    function atualizarFooter() {
        const footerElement = document.querySelector('.calendario-footer');
        if (dataSelecionada) {
            footerElement.classList.add('visivel');
            const dataFormatada = dataSelecionada.toLocaleDateString('pt-BR', { day: '2-digit', month: '2-digit' });
            textoDiaSelecionado.textContent = `Dia ${dataFormatada} selecionado`;
            botaoComprar.classList.remove('disabled');
            botaoComprar.href = `${urlBaseIngresso}${dataSelecionada.toISOString().split('T')[0]}`;
        } else {
            footerElement.classList.remove('visivel');
            textoDiaSelecionado.textContent = 'Nenhuma data selecionada';
            botaoComprar.classList.add('disabled');
            botaoComprar.href = '#';
        }
    }

    /**
     * Lida com o clique em um dia válido.
     */
    function selecionarDia(dataString) {
        const selecionadoAnterior = document.querySelector('.calendario-dia.selecionado');
        if (selecionadoAnterior) {
            selecionadoAnterior.classList.remove('selecionado');
        }
        
        const novoSelecionado = document.querySelector(`.calendario-dia[data-data="${dataString}"]`);
        if (novoSelecionado) {
            novoSelecionado.classList.add('selecionado');
        }
        
        dataSelecionada = new Date(dataString + 'T00:00:00');
        atualizarFooter();
    }

    /**
     * Adiciona os listeners para os botões de avançar/voltar mês.
     */
    function adicionarEventListenersNavegacao() {
        document.getElementById('prev-month').addEventListener('click', () => {
            dataAtual.setMonth(dataAtual.getMonth() - 1);
            renderizarCalendario();
        });
        document.getElementById('next-month').addEventListener('click', () => {
            dataAtual.setMonth(dataAtual.getMonth() + 1);
            renderizarCalendario();
        });
    }

    /**
     * A função principal que desenha o calendário na tela.
     */
    function renderizarCalendario() {
        const mes = dataAtual.getMonth();
        const ano = dataAtual.getFullYear();
        container.innerHTML = '';
        
        const header = document.createElement('div');
        header.className = 'calendario-header';
        header.innerHTML = `
            <button class="nav-btn" id="prev-month">&lt;</button>
            <h2>${new Date(ano, mes).toLocaleString('pt-BR', { month: 'long', year: 'numeric' })}</h2>
            <button class="nav-btn" id="next-month">&gt;</button>
        `;
        container.appendChild(header);

        const grid = document.createElement('div');
        grid.className = 'calendario-grid';
        const diasDaSemana = ['Dom', 'Seg', 'Ter', 'Qua', 'Qui', 'Sex', 'Sáb'];
        diasDaSemana.forEach(dia => grid.appendChild(Object.assign(document.createElement('div'), { className: 'dia-semana', textContent: dia })));
        
        const primeiroDiaDoMes = new Date(ano, mes, 1).getDay();
        for (let i = 0; i < primeiroDiaDoMes; i++) grid.appendChild(document.createElement('div'));

        const diasNoMes = new Date(ano, mes + 1, 0).getDate();
        for (let i = 1; i <= diasNoMes; i++) {
            const diaEl = document.createElement('div');
            diaEl.className = 'calendario-dia';
            const dataCorrente = new Date(ano, mes, i);
            const isPassado = dataCorrente < hoje;
            
            diaEl.innerHTML = `<span class="dia-numero">${i}</span><span class="dia-preco">...</span>`;

            if (isPassado) {
                diaEl.classList.add('desabilitado');
                diaEl.querySelector('.dia-preco').textContent = '';
            } else {
                const dataString = dataCorrente.toISOString().split('T')[0];
                diaEl.dataset.data = dataString;
                buscarEExibirPreco(diaEl, dataString);
            }

            if (dataSelecionada && dataCorrente.getTime() === dataSelecionada.getTime()) {
                diaEl.classList.add('selecionado');
            }
            grid.appendChild(diaEl);
        }
        container.appendChild(grid);
        adicionarEventListenersNavegacao();
    }

    // 4. INICIALIZAÇÃO
    // Aplica as cores e chama a primeira renderização.
    wrapper.style.setProperty('--cor-principal', calendarioConfig.corPrincipal);
    wrapper.style.setProperty('--cor-precos', calendarioConfig.corPrecos);
    renderizarCalendario();
});